<?php

namespace App\Http\Controllers\API;

use App\Basecolor;
use App\Colorant;
use App\Fandeck;
use App\Http\Controllers\Controller;
use App\ProductDetail;
use App\ProductFormulation;
use App\Shadecolor;
use App\TaxMetaInfo;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;

class ColorCodeController extends Controller
{
    public function getColor()
    {
        ini_set('memory_limit', '-1');

        $metainfo = TaxMetaInfo::first();
        $colorant = Colorant::get();
        // Apply mark_up_percentage and vat_percentage to each colorant's unitprice
        if ($metainfo) {
            $markUp = floatval($metainfo->mark_up_percentage ?? 0);
            $vat = floatval($metainfo->vat_percentage ?? 0);
            foreach ($colorant as $c) {
                $basePrice = floatval($c->unitPrice);
                $priceWithMarkup = $basePrice + ($basePrice * $markUp / 100);
                $finalPrice = $priceWithMarkup + ($priceWithMarkup * $vat / 100);
                $c->unitPrice = (string) round($finalPrice, 2);
            }
        }
        $base = Basecolor::select('id', 'base')->get();
        $products = ProductDetail::whereNotNull('name')->orderBy('order_index', 'asc')->get();

        $existingColumns = [];

        foreach ($products as $product) {
            $existingColumns[] = $product->code;
        }

        $selectFandeckColumns = array_merge(['id', 'name', 'order_index'], $existingColumns);
        $selectShadeColumns = array_merge(['id', 'colorcode', 'colorname', 'rvalue', 'gvalue', 'bvalue'], $existingColumns);

        $fandeck = Fandeck::select($selectFandeckColumns)->orderBy('order_index', 'asc')->get();
        $shadecolors = Shadecolor::select($selectShadeColumns)->get();
        $productTables = ProductFormulation::get();

        return response()->json(array_merge([
            'colorant' => $colorant,
            'base' => $base,
            'shadecolors' => $shadecolors,
            'products' => $products,
            'fandeck' => $fandeck,
            'productTables' => $productTables
        ]));
    }

    public function getUpdatedPrice()
    {
        $metainfo = TaxMetaInfo::first();
        $colorant = Colorant::get();
        $base = Basecolor::select('id', 'base')->get();
        $products = ProductDetail::whereNotNull('name')->orderBy('order_index', 'asc')->get();

        // Apply mark_up_percentage and vat_percentage to each colorant's unitprice
        if ($metainfo) {
            $markUp = floatval($metainfo->mark_up_percentage ?? 0);
            $vat = floatval($metainfo->vat_percentage ?? 0);
            foreach ($colorant as $c) {
                $basePrice = floatval($c->unitPrice);
                $priceWithMarkup = $basePrice + ($basePrice * $markUp / 100);
                $finalPrice = $priceWithMarkup + ($priceWithMarkup * $vat / 100);
                $c->unitPrice = (string) round($finalPrice, 2);
            }
        }

        return response()->json([
            'colorant' => $colorant,
            'base' => $base,
            'products' => $products,
        ]);
    }

    public function similarShadeColors(Request $request)
    {
        $r = (int) $request->input('rvalue', 0);
        $g = (int) $request->input('gvalue', 0);
        $b = (int) $request->input('bvalue', 0);
        $variation = (int) $request->input('variation', 20);

        $r = max(0, min(255, $r));
        $g = max(0, min(255, $g));
        $b = max(0, min(255, $b));
        $variation = max(0, min(255, $variation));

        $distanceExpr = "((rvalue - {$r})*(rvalue - {$r}) + (gvalue - {$g})*(gvalue - {$g}) + (bvalue - {$b})*(bvalue - {$b}))";

        $shadecolors = Shadecolor::select('*', DB::raw("{$distanceExpr} as distance_sq"))
            ->whereRaw('ABS(rvalue - ?) <= ?', [$r, $variation])
            ->whereRaw('ABS(gvalue - ?) <= ?', [$g, $variation])
            ->whereRaw('ABS(bvalue - ?) <= ?', [$b, $variation])
            ->orderBy('distance_sq', 'asc')
            ->get();


        return response()->json([
            'count' => $shadecolors->count(),
            'results' => $shadecolors,
        ]);
    }

    protected $unitConversion = [
        'feet'  => 1,
        'inch'  => 1 / 12,
        'meter' => 3.28084,
        'cm'    => 0.0328084,
    ];

    protected function toFeet(float $value, string $unit): float
    {
        if (!isset($this->unitConversion[$unit])) {
            throw new \InvalidArgumentException("Unsupported unit: $unit");
        }
        return $value * $this->unitConversion[$unit];
    }

    public function calculator(Request $request)
    {
        $validated = $request->validate([
            'product_id' => 'required|integer|exists:ms_product_details,id',
            'room' => 'required|array',
            'room.width'  => 'required|numeric|min:0',
            'room.height' => 'required|numeric|min:0',
            'room.unit'   => 'required|string|in:feet,inch,cm,meter',

            'doors' => 'sometimes|array',
            'doors.*.width'  => 'required_with:doors|numeric|min:0',
            'doors.*.height' => 'required_with:doors|numeric|min:0',
            'doors.*.unit'   => 'required_with:doors|string|in:feet,inch,cm,meter',

            'windows' => 'sometimes|array',
            'windows.*.width'  => 'required_with:windows|numeric|min:0',
            'windows.*.height' => 'required_with:windows|numeric|min:0',
            'windows.*.unit'   => 'required_with:windows|string|in:feet,inch,cm,meter',

            'coverageFt2PerGallon' => 'sometimes|numeric|min:50|max:1000',
        ]);

        $product = ProductDetail::find($validated['product_id']);

        if (!$product) {
            return response()->json(['error' => 'Product not found'], 404);
        }

        //assign default coverage if not provided
        $room = $validated['room'] ?? [];
        $doors = $validated['doors'] ?? [];
        $windows = $validated['windows'] ?? [];
        $coverageFt2PerGallon = $product->coverageFt2PerGallon ?? 0;

        // Convert room dimensions to feet
        $width  = $this->toFeet($room['width'], $room['unit']);
        $height = $this->toFeet($room['height'], $room['unit']);

        // Step 1: Total wall area
        $wallArea = $height * $width;

        // Step 2: Deduct doors
        $doorArea = 0;
        foreach ($doors as $door) {
            $doorWidth  = $this->toFeet($door['width'], $door['unit']);
            $doorHeight = $this->toFeet($door['height'], $door['unit']);
            $doorArea += $doorWidth * $doorHeight;
        }

        // Step 3: Deduct windows
        $windowArea = 0;
        foreach ($windows as $window) {
            $winWidth  = $this->toFeet($window['width'], $window['unit']);
            $winHeight = $this->toFeet($window['height'], $window['unit']);
            $windowArea += $winWidth * $winHeight;
        }

        // Step 4: Net paintable area
        $paintableArea = max($wallArea - ($doorArea + $windowArea), 0);

        // Step 5: Paint requirement
        $gallonsRequired = $paintableArea / $coverageFt2PerGallon;
        // Convert gallons to liters (1 gallon ≈ 3.78541 L)
        $litersRequired = $gallonsRequired * 3.78541;

        return [
            'total_wall_area_sqft' => round($wallArea, 2),
            'deductions_sqft' => [
                'doors'   => round($doorArea, 2),
                'windows' => round($windowArea, 2),
            ],
            'paintable_area_sqft' => round($paintableArea, 2),
            'paint_required' => [
                'gallons' => ceil($gallonsRequired * 100) / 100, // keep 2 decimals
                'liters'  => ceil($litersRequired * 100) / 100,
            ],
        ];
    }
}
