<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Client;
use App\Color;
use Illuminate\support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ColorController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        $userId = $user->id;

        $search = request('search');

        $colorQuery = Client::with([
            'color.productFormulation.fandeck',
            'color.productFormulation.product',
            'color.productFormulation.shade',
            'color.productFormulation.base'
        ])
            ->where('dealerid', $userId);

        if ($search) {
            $colorQuery = $colorQuery->where(function ($query) use ($search) {
                $query
                    ->orWhere('name', 'like', "%$search%")
                    ->orWhere('contact', 'like', "%$search%")
                    ->orWhereHas('color.productFormulation.shade', function ($q) use ($search) {
                        $q->where('colorname', 'like', "%$search%")
                            ->orWhere('colorcode', 'like', "%$search%");
                    })
                    ->orWhereHas('color.productFormulation.product', function ($q) use ($search) {
                        $q->where('name', 'like', "%$search%");
                    })
                    ->orWhereHas('color.productFormulation.fandeck', function ($q) use ($search) {
                        $q->where('name', 'like', "%$search%");
                    })
                    ->orWhereHas('color.productFormulation.base', function ($q) use ($search) {
                        $q->where('name', 'like', "%$search%");
                    });
            });
        }

        $colorinfo = $colorQuery->orderBy('id', 'DESC')->paginate(20);

        return response()->json([
            'status' => 'success',
            'data' => $colorinfo
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'address' => 'required|string|max:255',
            'contact' => 'required|string|max:255',
            'color_id' => 'required|integer',
            'size' => 'required|string|max:255',
            'quantity' => 'required|integer',
        ]);

        $dealerUser = auth()->user();
        $dealer = $dealerUser->dealer;

        $client = Client::firstOrCreate(
            [
                'contact' => $validated['contact'],
                'dealerid' => $dealer->id,
            ],
            [
                'name' => $validated['name'],
                'address' => $validated['address'],
                'status' => 0,
            ]
        );

        $color = Color::where([
            ['client_id', '=', $client->id],
            ['color_id', '=', $validated['color_id']],
            ['size', '=', $validated['size']],
        ])->first();

        if ($color) {
            $color->increment('quantity', $validated['quantity']);
            $color->update(['status' => 0]);
        } else {
            Color::create([
                'client_id' => $client->id,
                'color_id' => $validated['color_id'],
                'size' => $validated['size'],
                'quantity' => $validated['quantity'],
                'status' => 0,
            ]);
        }

        $colorInfo = Client::with([
            'color.productFormulation.fandeck',
            'color.productFormulation.product',
            'color.productFormulation.shade',
            'color.productFormulation.base'
        ])
            ->where('dealerid', $dealer->id)
            ->latest('id')
            ->paginate(20);

        return response()->json([
            'status' => 'success',
            'data' => $colorInfo
        ]);
    }


    public function clientdestroy($id)
    {
        Color::where('client_id', $id)->delete();
        Client::findOrFail($id)->delete();

        $colorinfo = Client::with([
            'color.productFormulation.fandeck',
            'color.productFormulation.product',
            'color.productFormulation.shade',
            'color.productFormulation.base'
        ])
            ->where('dealerid', Auth::user()->id)
            ->paginate(20);
        return response()->json([
            'status' => 'success',
            'data' => $colorinfo
        ]);
    }

    public function colordestroy($id)
    {
        Color::findOrFail($id)->delete();

        $colorinfo = Client::with([
            'color.productFormulation.fandeck',
            'color.productFormulation.product',
            'color.productFormulation.shade',
            'color.productFormulation.base'
        ])
            ->where('dealerid', Auth::user()->id)
            ->paginate(20);

        return response()->json([
            'status' => 'success',
            'data' => $colorinfo
        ]);
    }
}
