<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Order;
use App\Orderitem;
use App\Productimage;
use App\Cart;
use App\DealerInfo;
use App\ProductSize;
use App\DealerDiscount;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;

class OrderController extends Controller
{
    public function orderHistory(Request $request)
    {
        $sum = 0;
        $all = array();
        $user = Auth::user();
        $id = $user->id;

        $orderHistory = DB::table('orders')->select('id', 'dealer_code', 'user_id', 'date', 'status', 'remarks')->orderBy('created_at', 'DESC')->where('user_id', '=', $id)->paginate(15);

        foreach ($orderHistory as $order) {

            $test = DB::table('orderitems')
                ->join('product_size', 'orderitems.items_id', 'product_size.id')
                ->join('products', 'product_size.product_id', 'products.id')
                ->join('sizes', 'product_size.size_id', 'sizes.id')
                ->select('orderitems.id', 'products.id as pid', 'orderitems.order_id', 'orderitems.items_id', 'products.name', 'orderitems.quantity', 'product_size.price', 'product_size.size_id', 'product_size.product_id', 'sizes.size')
                ->where('orderitems.order_id', $order->id)
                ->get();
            $count = count($test);
            foreach ($test as $t) {
                $image = Productimage::where('product_id', $t->pid)->select('image', 'thumbnail')->get();
                $sum = $sum + ($t->price * $t->quantity);
                $t->image = $image;
            }
            $order->total_price = $sum;
            $order->order = $test;
            $sum = 0;
        }
        return response()->json($orderHistory);
    }

    public function editOrder($id, Request $request)
    {
        $this->validate($request, [
            'remarks' => 'required',
        ]);
        $order = Order::findOrFail($id);

        if ($order->status !== 'pending') {
            return response()->json([
                'status' => 'error',
                'message' => 'You cannot cancel this order because your order has been changed.'
            ], 400);
        }

        $order->status = 'cancelled';
        $order->remarks = $request->remarks;
        $order->save();

        $items = DB::table('orderitems')
            ->join('product_size', 'orderitems.items_id', '=', 'product_size.id')
            ->join('products', 'product_size.product_id', '=', 'products.id')
            ->join('sizes', 'product_size.size_id', '=', 'sizes.id')
            ->select(
                'orderitems.id',
                'products.id as pid',
                'orderitems.order_id',
                'orderitems.items_id',
                'products.name',
                'orderitems.quantity',
                'product_size.price',
                'product_size.size_id',
                'product_size.product_id',
                'sizes.size'
            )
            ->where('orderitems.order_id', $order->id)
            ->get();

        $productIds = $items->pluck('pid')->unique()->all();
        $imagesByProduct = Productimage::whereIn('product_id', $productIds)
            ->select('product_id', 'image', 'thumbnail')
            ->get()
            ->groupBy('product_id');

        $lis = [];
        foreach ($items as $t) {
            $image = $imagesByProduct->get($t->pid, collect());
            $lis[] = [
                'id' => $t->id,
                'order_id' => $t->order_id,
                'items_id' => $t->items_id,
                'name' => $t->name,
                'quantity' => $t->quantity,
                'price' => $t->price,
                'size_id' => $t->size_id,
                'product_id' => $t->product_id,
                'size' => $t->size,
                'image' => $image->values(),
            ];
        }

        $list = [
            'order_id' => $order->id,
            'dealer_code' => $order->dealer_code,
            'user_id' => $order->user_id,
            'date' => $order->date,
            'status' => $order->status,
            'remarks' => $order->remarks,
            'items' => $lis,
        ];
        return response()->json([
            'status' => 'success',
            'data' => $list
        ]);
    }

    public function cancelOrderItem($id)
    {
        try {
            $orderItem = Orderitem::findOrFail($id);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Order item not found.'
            ], 404);
        }

        try {
            $order = Order::findOrFail($orderItem->order_id);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Order not found.'
            ], 404);
        }

        if ($order->status !== 'pending') {
            return response()->json([
                'status' => 'error',
                'message' => 'You cannot cancel items for this order because its status has changed.'
            ], 400);
        }

        $allOrderItems = Orderitem::where('order_id', $orderItem->order_id)->get();

        if ($allOrderItems->count() === 1) {
            $order->status = 'cancelled';
            $order->save();
        } else {
            $orderItem->delete();
        }

        $items = DB::table('orderitems')
            ->join('product_size', 'orderitems.items_id', '=', 'product_size.id')
            ->join('products', 'product_size.product_id', '=', 'products.id')
            ->join('sizes', 'product_size.size_id', '=', 'sizes.id')
            ->select(
                'orderitems.id',
                'products.id as pid',
                'orderitems.order_id',
                'orderitems.status',
                'orderitems.items_id',
                'products.name',
                'orderitems.quantity',
                'product_size.price',
                'product_size.size_id',
                'product_size.product_id',
                'sizes.size'
            )
            ->where('orderitems.order_id', $order->id)
            ->get();

        $productIds = $items->pluck('pid')->unique()->all();
        $imagesByProduct = Productimage::whereIn('product_id', $productIds)
            ->select('product_id', 'image', 'thumbnail')
            ->get()
            ->groupBy('product_id');

        $lis = [];
        foreach ($items as $t) {
            $image = $imagesByProduct->get($t->pid, collect());
            $lis[] = [
                'id' => $t->id,
                'order_id' => $t->order_id,
                'items_id' => $t->items_id,
                'name' => $t->name,
                'quantity' => $t->quantity,
                'price' => $t->price,
                'size_id' => $t->size_id,
                'status' => $t->status ?? null,
                'product_id' => $t->product_id,
                'size' => $t->size,
                'image' => $image->values(),
            ];
        }

        $list = [
            'order_id' => $order->id,
            'dealer_code' => $order->dealer_code,
            'user_id' => $order->user_id,
            'date' => $order->date,
            'status' => $order->status,
            'remarks' => $order->remarks,
            'items' => $lis,
        ];
        return response()->json([
            'status' => 'success',
            'data' => $list
        ]);
    }

    public function store(Request $request)
    {
        $validated = Validator::make($request->all(), [
            'orders' => 'required|array',
            'orders.*.cart_id' => 'required|exists:carts,id'
        ]);

        if ($validated->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => $validated->errors()
            ], 422);
        }

        $user = Auth::user()->load('dealer');
        $due_amt = $user->dealer->due_amount;
        $dealerId = $user->dealer_id;
        $orderDate = now()->toDateString();

        $cartIds = collect($request->orders)->pluck('cart_id')->toArray();
        try {
            $carts = Cart::whereIn('id', $cartIds)->get()->keyBy('id');
            $totalCartPrice = $carts->sum('price');
            $totalPrice = $totalCartPrice + $due_amt;

            if ($user->dealer->order_limit < $totalPrice) {
                return response()->json([
                    'status' => 'error',
                    'message' => "The dealer order limit is Rs.{$user->dealer->order_limit}, but the current order amount is {$totalCartPrice} and due amount is {$due_amt}. Kindly adjust the order accordingly."
                ], 400);
            }

            $itemIds = $carts->pluck('item_id')->unique();
            $productSizes = ProductSize::with('product')->whereIn('id', $itemIds)->get()->keyBy('id');

            $categoryIds = $productSizes->pluck('product.category_id')->unique();
            $discounts = DealerDiscount::where('dealer_id', $dealerId)
                ->whereIn('category_id', $categoryIds)
                ->get()
                ->keyBy('category_id');

            $order = Order::create([
                'dealer_code' => $dealerId,
                'user_id' => $user->id,
                'date' => $orderDate,
                'status' => 'pending',
            ]);

            $totalAmount = 0;
            $orderItems = [];

            foreach ($request->orders as $item) {
                $cart = $carts[$item['cart_id']] ?? null;

                if (!$cart) {
                    continue;
                }

                $productSize = $productSizes[$cart->item_id] ?? null;
                if (!$productSize) {
                    continue;
                }

                $pricePerItem = $productSize->price;
                $quantity = $cart->quantity;
                $itemTotal = $quantity * $pricePerItem;
                $discountAmount = 0;
                $discountRate = 0;

                $categoryId = $productSize->product->category_id;
                $discount = $discounts[$categoryId] ?? null;

                if ($discount) {
                    $discountRate = $discount->discount;
                    $discountAmount = ($itemTotal * $discountRate) / 100;
                }

                $orderItems[] = new Orderitem([
                    'order_id' => $order->id,
                    'items_id' => $cart->item_id,
                    'quantity' => $quantity,
                    'items_price_rate' => $pricePerItem,
                    'price' => $itemTotal,
                    'discount_rate' => $discountRate,
                    'discount_amount' => $discountAmount,
                    'updated_quantity' => $quantity,
                ]);

                $totalAmount += $itemTotal;

                $cart->delete();
            }

            $order->items()->saveMany($orderItems);

            $dealer = DealerInfo::find($dealerId);
            if ($dealer) {
                $dealer->due_amount += $totalAmount;
                $dealer->save();
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }

        return response()->json([
            'message' => 'Success',
            'order_id' => $order->id
        ]);
    }
}
