<?php


namespace App\Http\Controllers;

use App\Client;
use App\DealerInfo;
use App\ImageAsset;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;

class ImageAssetController extends Controller
{
    public function index()
    {
        $data['imageAssets'] = ImageAsset::get();
        return view('dashboards.housePreview.index', $data);
    }

    public function create()
    {
        $data['dealers'] = DealerInfo::select('id', 'firm_name')->get();
        $data['clients'] = Client::select('id', 'name', 'contact')->get();
        return view('dashboards.housePreview.create', $data);
    }

    public function store(Request $request)
    {
        $request->validate([
            'dealer_id' => 'required|exists:dealer_info,id',
            'client_id' => 'nullable|exists:clients,id',
            'client_name' => 'required_without:client_id|string|max:255',
            'client_address' => 'required_without:client_id|string|max:255',
            'client_contact' => 'required_without:client_id|min:8|max:20|unique:clients,contact',
            'type' => 'required|in:Interior,Exterior',
            'original_file' => 'required|file|mimes:jpg,jpeg,png|max:2048',
            'svg_file' => 'required|file|mimes:svg|max:2048',
        ]);

        if (!empty($request->client_contact && ($request->client_id == null))) {
            $client = Client::Create([
                'dealerid' => $request->dealer_id,
                'contact' => $request->client_contact,
                'name' => $request->client_name,
                'address' => $request->client_address,
                'status' => 0,
            ]);
        }

        $clientId = $client->id ?? $request->client_id;

        if ($request->hasFile('svg_file') && $request->hasFile('original_file')) {

            //Store Original File
            $originalFile = $request->file('original_file');
            $originalFilename = 'clientOriginalHouse/' . uniqid('original_') . '.' . $originalFile->getClientOriginalExtension();
            $originalFile->storeAs('public', $originalFilename);

            //Store SVG File
            $svgFile = $request->file('svg_file');
            $filename = 'clientHouse/' . uniqid('svg_') . '.' . $svgFile->getClientOriginalExtension();
            $svgFile->storeAs('public', $filename);
        } else {
            return back()->withErrors(['Both files are required.'])->withInput();
        }

        $svgFile = $request->file('svg_file');
        $filename = 'clientHouse/' . uniqid('svg_') . '.' . $svgFile->getClientOriginalExtension();
        $svgFile->storeAs('public', $filename);

        ImageAsset::create([
            'dealer_id' => $request->dealer_id,
            'client_id' => $clientId,
            'type' => $request->type,
            'original_filename' => $originalFilename,
            'filename' => $filename,
        ]);

        return redirect()->route('house-previews.index')->with('success', 'House Preview created successfully.');
    }

    public function edit($id)
    {
        $data['dealers'] = DealerInfo::select('id', 'firm_name')->get();
        $data['clients'] = Client::select('id', 'name', 'contact')->get();
        $data['housePreview'] = ImageAsset::findOrFail($id);
        return view('dashboards.housePreview.edit', $data);
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'dealer_id' => 'required|exists:dealer_info,id',
            'client_id' => 'nullable|exists:clients,id',
            'client_name' => 'required_without:client_id|string|max:255',
            'client_address' => 'required_without:client_id|string|max:255',
            'client_contact' => 'required_without:client_id|min:8|max:20|unique:clients,contact,' . $id,
            'type' => 'required|in:Interior,Exterior',
            'original_file' => 'nullable|file|mimes:jpg,jpeg,png|max:2048',
            'svg_file' => 'nullable|file|mimes:svg|max:2048',
        ]);

        if (!empty($request->client_contact && ($request->client_id == null))) {
            $client = Client::Create([
                'dealerid' => $request->dealer_id,
                'contact' => $request->client_contact,
                'name' => $request->client_name,
                'address' => $request->client_address,
                'status' => 0,
            ]);
        }

        $clientId = $client->id ?? $request->client_id;

        $imageAsset = ImageAsset::findOrFail($id);
        $imageAsset->dealer_id = $request->dealer_id;
        $imageAsset->client_id = $clientId;
        $imageAsset->type = $request->type;

        //Store Original File
        if ($request->hasFile('original_file')) {
            if ($imageAsset->original_filename && Storage::disk('public')->exists($imageAsset->original_filename)) {
                Storage::disk('public')->delete($imageAsset->original_filename);
            }
            $originalFile = $request->file('original_file');
            $originalFilename = 'clientOriginalHouse/' . uniqid('original_') . '.' . $originalFile->getClientOriginalExtension();
            $originalFile->storeAs('public', $originalFilename);
            $imageAsset->original_filename = $originalFilename;
        }

        //Store SVG File
        if ($request->hasFile('svg_file')) {
            if ($imageAsset->filename && Storage::disk('public')->exists($imageAsset->filename)) {
                Storage::disk('public')->delete($imageAsset->filename);
            }
            $svgFile = $request->file('svg_file');
            $filename = 'clientHouse/' . uniqid('svg_') . '.' . $svgFile->getClientOriginalExtension();
            $svgFile->storeAs('public', $filename);
            $imageAsset->filename = $filename;
        }

        $imageAsset->save();

        return redirect()->route('house-previews.index')->with('success', 'House Preview updated successfully.');
    }

    public function destroy($id)
    {
        $imageAsset = ImageAsset::findOrFail($id);
        if ($imageAsset->filename && Storage::disk('public')->exists($imageAsset->filename)) {
            Storage::disk('public')->delete($imageAsset->filename);
        }
        $imageAsset->delete();
        return redirect()->route('house-previews.index')->with('success', 'House Preview deleted successfully.');
    }
}
